-- Create local functions to use. Stuff in pg_temp is session-local and disappears
-- at disconnection.

-- There are a few places where we test for existence and then do an IF NOT EXISTS.
-- The aim here is to a) give the best meaningful message, and we can only give
-- a message indicating that something was created if we looked for it first, and
-- b) we don't want to spam the server logs by omitting the IF NOT EXISTS, since that
-- will log errors to the server log, even if we catch them here. So we do both, and
-- will only show both our "thing created" message and the automatic NOTICE that we
-- typically get if something already existed in the event of a race condition, which
-- is fairly unlikely.

CREATE PROCEDURE pg_temp.create_extension(ename TEXT, target_schema TEXT, optional BOOLEAN = false) AS $proc$
BEGIN
  IF EXISTS (SELECT FROM pg_available_extensions WHERE name = ename) THEN
    IF NOT EXISTS (SELECT FROM pg_extension WHERE pg_extension.extname = ename) THEN
      BEGIN
        EXECUTE format('CREATE EXTENSION IF NOT EXISTS %I SCHEMA %I', quote_ident(ename), quote_ident(target_schema));
        RAISE NOTICE 'Installed Extension %', ename;
      EXCEPTION WHEN OTHERS THEN
        IF optional THEN
          RAISE WARNING 'Unable to install optional extension %', ename;
        ELSE
          RAISE EXCEPTION 'Unable to install required extension %', ename;
        END IF;
      END;
    ELSE
      RAISE NOTICE 'Extension % already installed', ename;
    END IF;
  ELSIF optional THEN
    RAISE NOTICE 'Optional extension % is not available', ename;
  ELSE
    RAISE EXCEPTION 'Required extension % is not available', ename;
  END IF;
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.create_schema(sname TEXT, owner TEXT) AS $proc$
BEGIN
  IF NOT EXISTS (SELECT FROM pg_namespace WHERE nspname = sname) THEN
    BEGIN
      EXECUTE format('CREATE SCHEMA IF NOT EXISTS %I', quote_ident(sname));
      EXECUTE format('ALTER SCHEMA %I OWNER TO %I', quote_ident(sname), quote_ident(owner));
      RAISE NOTICE 'Created schema %', sname;
    EXCEPTION WHEN duplicate_object THEN
      RAISE NOTICE 'Schema % already created', sname;
    END;
  ELSE
    RAISE NOTICE 'Schema % already created', sname;
  END IF;

  IF sname <> 'public' AND owner <> (SELECT nspowner::regrole::text FROM pg_namespace WHERE nspname = sname) THEN
    EXECUTE format('ALTER SCHEMA %I OWNER TO %I', quote_ident(sname), quote_ident(owner));
    RAISE NOTICE 'Altered schema % owner to %', sname, owner;
  END IF;
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.setup_app(appname TEXT, schema_suffix TEXT DEFAULT '') AS $proc$
DECLARE
  owner TEXT = 'ge_migrator';
  app_user TEXT = 'ge_app';
  schema_name TEXT = appname || schema_suffix;
  acting_user TEXT = (SELECT current_user);
  was_granted BOOLEAN;
  postgres_server_version_num NUMERIC;
BEGIN
  SELECT version FROM current_setting('server_version_num') as version INTO postgres_server_version_num;

  -- we need to grant schema ownership role to current user so we can assign that role as schema owner in RDS environments
  -- in pg16+ this requires the SET option on the owner role
  IF postgres_server_version_num >= 160000 THEN
    SELECT EXISTS (
      SELECT
      FROM pg_auth_members
      WHERE member::regrole = acting_user::regrole
        AND roleid::regrole = owner::regrole
        AND set_option
    ) INTO was_granted;
    IF NOT was_granted THEN
      EXECUTE format('GRANT %I TO %I WITH SET TRUE', owner, acting_user);
    END IF;
  ELSE
    SELECT EXISTS (
      SELECT
      FROM pg_auth_members
      WHERE member::regrole = acting_user::regrole
        AND roleid::regrole = owner::regrole
    ) INTO was_granted;
    IF NOT was_granted THEN
      EXECUTE format('GRANT %I TO %I', owner, acting_user);
    END IF;
  END IF;

  CALL pg_temp.create_schema(schema_name, owner);

  -- set role necessary in AWS Aurora as in some circumstances a db owner can't GRANT on a schema owned by another role
  EXECUTE format('SET ROLE TO %I', owner);
  EXECUTE format('GRANT ALL ON SCHEMA %I TO %I', quote_ident(schema_name), quote_ident(app_user));
  EXECUTE format('SET ROLE TO %I', acting_user);

  IF NOT was_granted THEN
    -- role is no longer needed
    EXECUTE format('REVOKE %I FROM %I', owner, acting_user);
  END IF;
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.create_extensions(target_schema TEXT DEFAULT 'public') AS $proc$
BEGIN
  RAISE NOTICE 'Creating extensions';
  CALL pg_temp.create_extension('pgcrypto', target_schema);
  CALL pg_temp.create_extension('intarray', target_schema);
  CALL pg_temp.create_extension('amcheck', target_schema, true);
  CALL pg_temp.create_extension('pg_trgm', target_schema);
  CALL pg_temp.create_extension('pgstattuple', target_schema, true);
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.setup_apps(schema_suffix TEXT DEFAULT '') AS $proc$
BEGIN
  RAISE NOTICE 'Creating schemas';
  CALL pg_temp.setup_app('admin', schema_suffix);
  CALL pg_temp.setup_app('build_cache', schema_suffix);
  CALL pg_temp.setup_app('build_scans', schema_suffix);
  CALL pg_temp.setup_app('keycloak', schema_suffix);
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.setup_privileged_functions(target_schema TEXT DEFAULT 'public') AS $proc$
DECLARE
  owner_role TEXT = 'ge_migrator';
  app_role TEXT = 'ge_app';
BEGIN
  RAISE NOTICE 'Setting up privileged functions';
  EXECUTE format($exec$
    CREATE OR REPLACE FUNCTION %I.setup_auto_explain(log_min_duration TEXT, log_format TEXT DEFAULT 'json', log_level TEXT DEFAULT 'info') RETURNS VOID AS $func$
    BEGIN
      BEGIN
        LOAD 'auto_explain';
      EXCEPTION WHEN OTHERS THEN
        RAISE WARNING SQLSTATE 'U0002' USING MESSAGE = 'Unable to load auto_explain: ' || SQLSTATE || ' ' || SQLERRM;
      END;
      PERFORM set_config('auto_explain.log_min_duration', log_min_duration, false);
      PERFORM set_config('auto_explain.log_format', log_format, false);
      BEGIN
        PERFORM set_config('auto_explain.log_level', log_level, false);
      EXCEPTION WHEN OTHERS THEN
        RAISE WARNING SQLSTATE 'U0002' USING MESSAGE = 'Unable to set auto_explain.log_level: ' || SQLSTATE || ' ' || SQLERRM;
      END;
    END;
    $func$ LANGUAGE plpgsql SECURITY DEFINER;
  $exec$, quote_ident(target_schema));

  EXECUTE format($exec$
    CREATE OR REPLACE FUNCTION %I.setup_local_nested_auto_explain(log_min_duration TEXT) RETURNS VOID AS $func$
    SELECT set_config('auto_explain.log_min_duration', log_min_duration, true);
    SELECT set_config('auto_explain.log_nested_statements', 'true', true);
    $func$ LANGUAGE SQL SECURITY DEFINER;
  $exec$, quote_ident(target_schema));

  BEGIN
    EXECUTE format($exec$
      GRANT EXECUTE ON FUNCTION %I.bt_index_check(regclass) TO ge_app;
    $exec$, quote_ident(target_schema));
  EXCEPTION WHEN OTHERS THEN
    -- GCS doesn't have amcheck available, but allows the CREATE EXTENSION command above
    -- to pass with a warning. This will fail though - just ignore it.
    RAISE WARNING 'Unable to grant permission on %.bt_index_check(regclass) - %', target_schema, SQLSTATE;
  END;

  EXECUTE format($exec$
    CREATE OR REPLACE FUNCTION ge_stat_activity() RETURNS SETOF pg_catalog.pg_stat_activity AS $func$
    SELECT * FROM pg_catalog.pg_stat_activity
    WHERE datname = current_database()
    $func$ LANGUAGE SQL SECURITY DEFINER;
  $exec$, quote_ident(target_schema));

  EXECUTE format($exec$
    CREATE OR REPLACE VIEW %I.ge_stat_activity AS
    SELECT * FROM ge_stat_activity();
  $exec$, quote_ident(target_schema));

  BEGIN
    EXECUTE format('GRANT SELECT ON %I.ge_stat_activity TO %I', quote_ident(target_schema), owner_role);
  EXCEPTION WHEN OTHERS THEN
    -- play it safe with grants
    RAISE WARNING 'Unable to grant permission on ge_stat_activity to % - %', owner_role, SQLSTATE;
  END;

  BEGIN
    EXECUTE format('GRANT SELECT ON %I.ge_stat_activity TO %I', quote_ident(target_schema), app_role);
  EXCEPTION WHEN OTHERS THEN
    -- play it safe with grants
    RAISE WARNING 'Unable to grant permission on ge_stat_activity to % - %', app_role, SQLSTATE;
  END;

  BEGIN
    EXECUTE format('GRANT pg_stat_scan_tables TO %I', app_role);
  EXCEPTION WHEN OTHERS THEN
    -- play it safe with grants
    RAISE WARNING 'Unable to grant pg_stat_scan_tables to % - %', app_role, SQLSTATE;
  END;

END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.setup_shared_schema(target_schema TEXT DEFAULT 'public') AS $proc$
BEGIN
  IF target_schema <> 'public' THEN
    RAISE NOTICE 'Setting up shared schema';
    CALL pg_temp.create_schema(target_schema, current_user);
    EXECUTE format('GRANT USAGE ON SCHEMA %I TO PUBLIC', quote_ident(target_schema));
  END IF;
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.setup(target_schema TEXT DEFAULT 'public') AS $proc$
BEGIN
  CALL pg_temp.setup_shared_schema(target_schema);

  RAISE NOTICE '---';
  CALL pg_temp.create_extensions(target_schema);

  RAISE NOTICE '---';
  CALL pg_temp.setup_apps();

  RAISE NOTICE '---';
  CALL pg_temp.setup_privileged_functions(target_schema);

  COMMIT;
  RAISE NOTICE '---';
  RAISE NOTICE 'Database objects set up successfully';
END;
$proc$
LANGUAGE plpgsql;

DO $$
BEGIN
  CALL pg_temp.setup();
END;
$$ LANGUAGE plpgsql;
