-- List of issues to report
CREATE TEMP TABLE db_setup_check_issues (
  id serial,
  message text NOT NULL,
  is_error boolean DEFAULT true
);

-- roles
DO $$
  DECLARE
    all_schemas CONSTANT text[] := ARRAY['admin', 'build_cache', 'build_scans', 'keycloak'];
    migrator_user CONSTANT text := 'ge_migrator';
    app_user CONSTANT text := 'ge_app';
    monitor_user CONSTANT text := 'ge_monitor';
    schema text;
    owner_role text;
    user_role text;
    user_role_found boolean;
    owner_role_found boolean;
  BEGIN
    PERFORM FROM pg_catalog.pg_roles WHERE pg_roles.rolname = migrator_user;
    IF NOT FOUND THEN
      INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s user does not exist', migrator_user));
    END IF;

    PERFORM FROM pg_catalog.pg_roles WHERE pg_roles.rolname = app_user;
    IF NOT FOUND THEN
      INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s user does not exist', app_user));
    END IF;

    PERFORM FROM pg_catalog.pg_roles WHERE pg_roles.rolname = monitor_user;
    IF NOT FOUND THEN
      INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s user does not exist', monitor_user));
    END IF;

    FOREACH schema IN ARRAY all_schemas LOOP
      owner_role := schema || '_owner';
      user_role := schema || '_user';

      PERFORM FROM pg_catalog.pg_roles WHERE pg_roles.rolname = owner_role;
      IF NOT FOUND THEN
        owner_role_found := false;
        INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s role does not exist', owner_role));
      ELSE
        owner_role_found := true;
        IF NOT pg_has_role(migrator_user, owner_role, 'MEMBER') THEN
          INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s role does not have the %s role granted', migrator_user, owner_role));
        END IF;
      END IF;

      PERFORM FROM pg_catalog.pg_roles WHERE pg_roles.rolname = user_role;
      IF NOT FOUND THEN
        user_role_found := false;
        INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s role does not exist', user_role));
      ELSE
        user_role_found := true;
        IF NOT pg_has_role(app_user, user_role, 'MEMBER') THEN
          INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s role does not have the %s role granted', app_user, user_role));
        END IF;
      END IF;

      -- special case for build scans to support triggering vacuums
      IF schema = 'build_scans' AND owner_role_found AND user_role_found THEN
        IF NOT pg_has_role(user_role, owner_role, 'USAGE') THEN
          INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s role does not have the %s role granted', user_role, owner_role));
        END IF;
      END IF;
    END LOOP;
  END;
$$
LANGUAGE plpgsql;

-- schemas
DO $$
  DECLARE
    all_schemas CONSTANT text[] := ARRAY['admin', 'build_cache', 'build_scans', 'keycloak'];
    shared_schema CONSTANT text := 'public'; -- NOTE: adjust if using a non-standard shared schema
    all_schema_privs CONSTANT text[] := ARRAY['USAGE', 'CREATE'];
    owner_role text;
    user_role text;
    schema text;
    priv text;
    role text;
  BEGIN
    FOREACH schema IN ARRAY all_schemas LOOP
      owner_role := schema || '_owner';
      user_role := schema || '_user';

      PERFORM FROM pg_catalog.pg_namespace WHERE pg_namespace.nspname = schema;
      IF NOT FOUND THEN
        INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s schema does not exist', schema));
      ELSE
        PERFORM FROM pg_namespace WHERE nspname = schema and nspowner = owner_role::regrole;
        IF NOT FOUND THEN
          INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s schema is not owned by the %s role', schema, owner_role));
        END IF;

        FOREACH role IN ARRAY ARRAY[owner_role, user_role] LOOP
          PERFORM FROM pg_catalog.pg_roles WHERE pg_roles.rolname = role;
          IF NOT FOUND THEN
            CONTINUE;
          END IF;

          FOREACH priv IN ARRAY all_schema_privs LOOP
            IF NOT has_schema_privilege(role, schema, priv) THEN
              INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s role does not have the %s privilege on the %s schema', role, priv, schema));
            END IF;
          END LOOP;
          IF NOT has_schema_privilege(role, shared_schema, 'USAGE') THEN
            INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s role does not have the %s privilege on the %s schema', role, 'USAGE', shared_schema));
          END IF;
        END LOOP;
      END IF;
    END LOOP;
  END;
$$
LANGUAGE plpgsql;

-- extensions
DO $$
  DECLARE
    required_extensions CONSTANT text[] := ARRAY['pgcrypto', 'intarray', 'pg_trgm'];
    optional_extensions CONSTANT text[] := ARRAY['amcheck', 'pgstattuple'];
    ext TEXT;
  BEGIN
    FOREACH ext IN ARRAY required_extensions LOOP
      PERFORM FROM pg_catalog.pg_extension WHERE extname = ext;
      IF NOT FOUND THEN
        INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The required extension %s is not present', ext));
      END IF;
    END LOOP;
    FOREACH ext IN ARRAY optional_extensions LOOP
      PERFORM FROM pg_catalog.pg_extension WHERE extname = ext;
      IF NOT FOUND THEN
        INSERT INTO pg_temp.db_setup_check_issues (message, is_error) VALUES (format('The optional extension %s is not present', ext), false);
      END IF;
    END LOOP;
  END;
$$ LANGUAGE plpgsql;

-- tables
DO $$
  DECLARE
    all_schemas CONSTANT text[] := ARRAY['admin', 'build_cache', 'build_scans', 'keycloak'];
    -- there is also MAINTAIN, but this is in pg17+ only, we currently rely on build_scans_owner being granted to
    -- build_scans_user to trigger vacuums manually
    owner_schema_privs CONSTANT text[] := ARRAY['SELECT', 'INSERT', 'UPDATE', 'DELETE', 'TRUNCATE', 'REFERENCES', 'TRIGGER'];
    user_schema_privs CONSTANT text[] := ARRAY['SELECT', 'INSERT', 'UPDATE', 'DELETE', 'TRUNCATE'];
    owner_role text;
    user_role text;
    schema text;
    table_name text;
    priv text;
BEGIN
  FOREACH schema IN ARRAY all_schemas LOOP
    PERFORM FROM pg_catalog.pg_namespace WHERE pg_namespace.nspname = schema;
    IF NOT FOUND THEN
      CONTINUE;
    END IF;

    owner_role := schema || '_owner';
    user_role := schema || '_user';

    PERFORM FROM pg_catalog.pg_roles WHERE pg_roles.rolname = owner_role;
    IF FOUND THEN
      FOR table_name IN SELECT relname FROM pg_class WHERE relnamespace = schema::regnamespace AND relkind not in ('i', 'S', 't', 'c', 'I') LOOP
        PERFORM FROM pg_class WHERE relnamespace = schema::regnamespace AND relname = table_name AND relowner = owner_role::pg_catalog.regrole;
        IF NOT FOUND THEN
            INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s role is not the owner of the %s table', owner_role, schema || '.' || table_name));
        END IF;
        FOREACH priv IN ARRAY owner_schema_privs LOOP
          IF NOT has_table_privilege(owner_role, schema || '.' || table_name, priv) THEN
            INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s role does not have the %s privilege on the %s table', owner_role, priv, schema || '.' || table_name));
          END IF;
        END LOOP;
      END LOOP;
    END IF;

    PERFORM FROM pg_catalog.pg_roles WHERE pg_roles.rolname = user_role;
    IF FOUND THEN
      FOR table_name IN SELECT relname FROM pg_class WHERE relnamespace = schema::regnamespace AND relkind not in ('i', 'S', 't', 'c', 'I') ORDER BY relname LOOP
        FOREACH priv IN ARRAY user_schema_privs LOOP
          IF NOT has_table_privilege(user_role, schema || '.' || table_name, priv) THEN
            INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s role does not have the %s privilege on the %s table', user_role, priv, schema || '.' || table_name));
          END IF;
        END LOOP;
      END LOOP;
    END IF;

  END LOOP;
END;
$$
LANGUAGE plpgsql;

-- tablespaces
CREATE FUNCTION pg_temp.role_default_tablespace(role text) RETURNS text AS $$
WITH joined_config as (
  SELECT unnest(rolconfig) AS setting
  FROM pg_roles
  WHERE rolname = role),
config AS (
  SELECT split_part(setting, '=', 1) AS name,
         split_part(setting, '=', 2) AS val
  FROM joined_config
)
SELECT val
FROM config
WHERE name = 'default_tablespace';
$$ LANGUAGE sql;

-- tablespaces
DO $$
DECLARE
  all_roles CONSTANT text[] := ARRAY['ge_migrator', 'ge_app', 'ge_monitor', 'admin_owner', 'admin_user', 'build_cache_owner', 'build_cache_user', 'build_scans_owner', 'build_scans_user', 'keycloak_owner', 'keycloak_user'];
  db_default_tablespace TEXT;
  user_default_tablespace TEXT;
  role TEXT;
BEGIN
  SELECT spcname INTO db_default_tablespace
  FROM pg_database
  JOIN pg_tablespace ON dattablespace = pg_tablespace.oid
  where datname = current_database();

  FOREACH role IN ARRAY all_roles LOOP
    PERFORM FROM pg_catalog.pg_roles WHERE pg_roles.rolname = role;
    IF NOT FOUND THEN
      CONTINUE;
    END IF;

    user_default_tablespace := pg_temp.role_default_tablespace(role);
    IF user_default_tablespace IS NOT NULL AND (user_default_tablespace IS DISTINCT FROM db_default_tablespace) AND NOT has_tablespace_privilege(role, user_default_tablespace, 'CREATE') THEN
      INSERT INTO pg_temp.db_setup_check_issues (message) VALUES (format('The %s role does not have the CREATE privilege on the %s tablespace, which is that role''s default', role, user_default_tablespace));
    END IF;
  END LOOP;
END;
$$ LANGUAGE plpgsql;

DO $$
DECLARE
  m TEXT;
  found_errors boolean;
BEGIN
  FOR m IN SELECT message FROM pg_temp.db_setup_check_issues LOOP
    RAISE WARNING '%', m;
  END LOOP;
  SELECT bool_or(is_error) INTO found_errors FROM pg_temp.db_setup_check_issues;
  IF found_errors THEN
    RAISE EXCEPTION 'Found errors in database setup';
  END IF;
END;
$$ LANGUAGE plpgsql;
