#!/usr/bin/env bash

set -e
set -o functrace
set -o pipefail

logger() {
    echo "$(date +'%Y-%m-%d %T') - $1";
}

print_help_and_exit() {
    echo "Usage: PGPASSWORD=\"xxx\" [DATABASE_APP_USER_PASSWORD=\"xxx\"] [DATABASE_MIGRATOR_USER_PASSWORD=\"xxx\"] [DATABASE_MONITOR_USER_PASSWORD=\"xxx\"] setup.sh --host=my-host.db.com [--port=5432] [--user=postgres] [--database=gradle_enterprise]"
    echo ""
    echo "  Sets up a user-managed database for Develocity"
    echo ""
    echo "   --host     Host where the user-managed database is located. Required."
    echo "   --port     Port to connect to. Default 5432."
    echo "   --user     Database user to connect as (must be superuser). Default 'postgres'."
    echo "   --database Database name to connect to. Default 'gradle_enterprise'."
    echo "   --role     Database role to connect as if different to the user. Optional."
    echo "   --iam-rds  Use this parameter if using AWS IAM authentication to connect to your user-managed database."
    echo ""
    echo "If app / migrator / monitor user password is provided, it will be set by this script. If not, you'll need to set it yourself."
    echo ""
    echo "   Examples:"
    echo "      PGPASSWORD=\"xxx\" setup.sh --host=my-host.db.com"
    echo ""
    exit "$1"
}

create_users() {
  logger "Creating database users"
  $PSQL_WITH_GE_DB -f "${SQL_DIR}/create_db_users${SQL_FILE_SUFFIX}.sql"
}

export PGAPPNAME=$(basename "$0")

DB_HOST=
DB_PORT=5432
DB_USER=postgres
DB_NAME=gradle_enterprise
DB_SHARED_SCHEMA=public
DB_ROLE=
IAM_RDS=
SQL_FILE_SUFFIX=
CREATE_USERS_FIRST=

# Parse and validate input
for i in "$@"
do
case ${i} in
    --host=*)
    DB_HOST="${i#*=}"
    ;;
    --port=*)
    DB_PORT="${i#*=}"
    ;;
    --user=*)
    DB_USER="${i#*=}"
    ;;
    --role=*)
    DB_ROLE="${i#*=}"
    ;;
    --database=*)
    DB_NAME="${i#*=}"
    ;;
    --shared-schema=*)
    DB_SHARED_SCHEMA="${i#*=}"
    ;;
    --iam-rds)
    IAM_RDS=YES
    ;;
    --no-schema-specific-roles)
    SQL_FILE_SUFFIX=_no_schema_roles
    CREATE_USERS_FIRST=YES
    ;;
    *)
	echo "ERROR: Unsupported Parameter ${i}"
	HELP=YES
	HELP_EXIT_CODE=1
    ;;
esac
done

# Print Usage
if [ -n "$HELP" ]; then
  print_help_and_exit ${HELP_EXIT_CODE}
fi

if [ -z "$DB_HOST" ] ; then
  logger "Please specify your database host via the --host parameter"
  print_help_and_exit 1
fi

if [ -n "$PGPASSWORD" ] ; then
  logger "Using password authentication"
else
  if [ -z "$IAM_RDS" ]; then
    logger "Please specify your database password via the PGPASSWORD environmental variable or specify --iam-rds if using AWS IAM authentication."
    print_help_and_exit 1
  fi

  logger "Using AWS IAM database authentication"
  if [[ $DB_HOST != *.rds.amazonaws.com ]] ; then
    logger "Unexpected database hostname for RDS IAM access - expected hostname in the form *.rds.amazonaws.com"
    exit 1
  fi

  if command -v aws >/dev/null; then
    # Extract AWS region from hostname
    REGION="$(echo -n "$DB_HOST" | sed "s/\\.rds\\.amazonaws\\.com$//" | sed s/^.*\\.//)"
    if [ -z "$REGION" ] ; then
      logger "Could not determine region from hostname $DB_HOST"
      exit 1
    fi
    logger "Computed AWS region $REGION from hostname $DB_HOST"
    if AWS_RDS_OUTPUT="$(aws rds generate-db-auth-token --hostname "$DB_HOST" --port "$DB_PORT" --username "$DB_USER" --region "$REGION" 2>&1)" ; then
      logger "Generated RDS auth token"
      export PGPASSWORD="$AWS_RDS_OUTPUT"
    else
      logger "Could not generate auth token. Error was:"
      echo "$AWS_RDS_OUTPUT"
      exit 1
    fi
  elif [ -n "$JAVA_HOME" ]; then
    if CREDENTIAL_GENERATOR_OUTPUT="$($JAVA_HOME/bin/java -jar /opt/lib/database-credentials-generator.jar --hostname "$DB_HOST" --port "$DB_PORT" --username "$DB_USER" 2>&1)" ; then
      logger "Generated RDS auth token"
      export PGPASSWORD="$CREDENTIAL_GENERATOR_OUTPUT"
    else
      logger "Could not generate auth token. Error was:"
      echo "$CREDENTIAL_GENERATOR_OUTPUT"
      exit 1
    fi
  else
    logger "No available method to generate database auth token. Expected the AWS CLI (\`aws\`) to be installed."
    exit 1
  fi

  # the default sslmode of "prefer" doesn't seem to work when connecting to RDS with rds_iam enabled
  export PGSSLMODE=require
fi

CURRENT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"
SQL_DIR="$CURRENT_DIR/sql"

CONN_PARAMS="-U $DB_USER -h $DB_HOST -p $DB_PORT"
PSQL_PARAMS="$CONN_PARAMS -v ON_ERROR_STOP=1"
PSQL_WITH_GE_DB="psql $PSQL_PARAMS -d $DB_NAME"

if [ -n "$DB_ROLE" ] ; then
  if [ -n "$PGOPTIONS" ] ; then
    PGOPTIONS="$PGOPTIONS "
  fi
  export PGOPTIONS="${PGOPTIONS}-c role=$DB_ROLE"
fi

logger "Setting up database"

# Sanity check - can we connect to the postgres db as the given user?
logger "Testing database connection"
sanity_check_output=$($PSQL_WITH_GE_DB -t -c "SELECT 1234" 2>&1 | xargs echo -n) || true
if [[ $sanity_check_output != "1234" ]] ; then
  logger "Could not connect to the $DB_NAME database at $DB_HOST $DB_PORT: $sanity_check_output"
  exit 1
fi

if [ -n "$CREATE_USERS_FIRST" ] ; then
  create_users
fi

logger "Creating top-level database objects"
if [[ $DB_SHARED_SCHEMA != "public" ]] ; then
  cat "${SQL_DIR}/create_db_objects${SQL_FILE_SUFFIX}.sql" \
    | sed "s/CALL pg_temp.setup();/CALL pg_temp.setup('$DB_SHARED_SCHEMA');/" \
    > /tmp/create_db_objects_non_public_shared_schema.sql
  $PSQL_WITH_GE_DB -f /tmp/create_db_objects_non_public_shared_schema.sql
else
  $PSQL_WITH_GE_DB -f "${SQL_DIR}/create_db_objects${SQL_FILE_SUFFIX}.sql"
fi

if [ -z "$CREATE_USERS_FIRST" ] ; then
  create_users
fi

if [ -n "$DATABASE_APP_USER_PASSWORD" ]; then
  logger "Setting app user password"
  $PSQL_WITH_GE_DB -c "ALTER USER ge_app PASSWORD '$DATABASE_APP_USER_PASSWORD'"
fi

if [ -n "$DATABASE_MIGRATOR_USER_PASSWORD" ]; then
  logger "Setting migrator user password"
  $PSQL_WITH_GE_DB -c "ALTER USER ge_migrator PASSWORD '$DATABASE_MIGRATOR_USER_PASSWORD'"
fi

if [ -n "$DATABASE_MONITOR_USER_PASSWORD" ]; then
  logger "Setting monitor user password"
  $PSQL_WITH_GE_DB -c "ALTER USER ge_monitor PASSWORD '$DATABASE_MONITOR_USER_PASSWORD'"
fi

if [ -n "$IAM_RDS" ]; then
  logger "Granting rds_iam role to database users"
  $PSQL_WITH_GE_DB -c "GRANT rds_iam TO ge_app, ge_migrator"
fi

logger "Database setup complete"
