#!/bin/bash

# Set default values for optional parameters
apply=false
destroy=false
dashboard_folder_path=""
grafana_url=$(terraform -chdir="stages/provision-aws-grafana" output grafana_url)
grafana_token=$(terraform -chdir="stages/provision-aws-grafana" output grafana_token)
datasource_id=$(terraform -chdir="stages/provision-aws-grafana-datasource" output datasource_id)

# Parse the command line arguments
while [[ $# -gt 0 ]]; do
  case $1 in
    --destroy)
      destroy=true
      shift # past argument
      ;;
    --apply)
      apply=true
      shift # past argument
      ;;
    --dashboard-folder-path)
      dashboard_folder_path=$2
      shift # past argument
      shift # past value
      ;;
    --grafana-url)
      grafana_url=$2
      shift # past argument
      shift # past value
      ;;
    --grafana-token)
      grafana_token=$2
      shift # past argument
      shift # past value
      ;;
    --datasource-id)
      datasource_id=$2
      shift # past argument
      shift # past value
      ;;
    *)
      echo "Unknown argument: $1"
      exit 1
      ;;
  esac
done

# Check for required parameters
if [[ -z "$dashboard_folder_path" ]]; then
  echo "Error: --dashboard-folder-path is required"
  exit 1
fi

if [[ -z "$datasource_id" ]]; then
  echo "Error: --datasource-id is required, this is normally provided by the terraform output of the preceding stage"
  exit 1
fi


if [[ -z "$grafana_token" ]]; then
  echo "Error: --grafana-token is required, this is normally provided by the terraform output of the preceding stage"
  exit 1
fi

if [[ -z "$grafana_url" ]]; then
  echo "Error: --grafana-url is required, this is normally provided by the terraform output of the preceding stage"
  exit 1
fi

if [[ $dashboard_folder_path != /* ]]; then
      if ! command -v realpath &> /dev/null
      then
          echo "The dashboard_folder_path was a relative path and realpath is not installed. Please provide an absolute path."
          exit 1
      else
          dashboard_folder_path=$(realpath -s $dashboard_folder_path)
      fi
fi


terraform -chdir="stages/provision-drv-dashboards" init

if [[ $destroy == true ]]; then
  # Call terraform with the variables
  terraform -chdir="stages/provision-aws-grafana" destroy \
    -var "dashboard_folder_path=$dashboard_folder_path" \
    -var "datasource_id=$datasource_id" \
    -var "grafana_url=$grafana_url" \
    -var "grafana_token=$grafana_token"
fi

if [[ $apply == true ]]; then
  # Call terraform with the variables
  terraform -chdir="stages/provision-drv-dashboards" apply \
    -var "dashboard_folder_path=$dashboard_folder_path" \
    -var "datasource_id=$datasource_id" \
    -var "grafana_url=$grafana_url" \
    -var "grafana_token=$grafana_token"
else
  echo "Running terraform plan..."
  echo "If you want to apply the changes, run this script with the --apply flag"

  # Call terraform with the variables
  terraform -chdir="stages/provision-drv-dashboards" plan \
    -var "dashboard_folder_path=$dashboard_folder_path" \
    -var "datasource_id=$datasource_id" \
    -var "grafana_url=$grafana_url" \
    -var "grafana_token=$grafana_token"
fi
