-- The create_db_objects.sql script must be run before this one.

-- To alter the specific db users set up, edit the final DO block in the file.

-- Does NOT attempt to create credentials for the created users - this is left to customers to do

CREATE PROCEDURE pg_temp.verify_required_roles() AS $proc$
DECLARE
  app TEXT;
  role TEXT;
BEGIN
  RAISE NOTICE '---';
  RAISE NOTICE 'Verifying required roles';
  FOREACH app in ARRAY ARRAY['admin', 'build_cache', 'build_scans', 'keycloak'] LOOP
    FOREACH role in ARRAY ARRAY['owner', 'user'] LOOP
      IF NOT EXISTS (SELECT FROM pg_roles WHERE rolname = app || '_' || role) THEN
        RAISE EXCEPTION 'Expected role %_% to exist, but it did not. Please run the create_db_objects.sql script first', app, role;
      END IF;
    END LOOP;
  END LOOP;
END;
$proc$ LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.grant_role_to_user(rolename TEXT, username TEXT) AS $proc$
DECLARE
  r TEXT;
BEGIN
  IF NOT EXISTS (SELECT FROM pg_auth_members WHERE roleid = rolename::regrole AND member = username::regrole) THEN
    EXECUTE format('GRANT %I TO %I', quote_ident(rolename), quote_ident(username));
    RAISE NOTICE 'Granted % to %', rolename, username;
  ELSE
    RAISE NOTICE 'User % already has role %', username, rolename;
  END IF;
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.make_user(username TEXT) AS $proc$
BEGIN
  IF NOT EXISTS (SELECT FROM pg_roles WHERE rolname = username) THEN
    BEGIN
      EXECUTE format('CREATE USER %I', username);
      RAISE NOTICE 'Created user %', username;
    EXCEPTION WHEN duplicate_object THEN
      RAISE DEBUG 'User % already exists', username;
    END;
  ELSE
    RAISE NOTICE 'User % already exists', username;
  END IF;
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.setup_user(username TEXT, suffix TEXT, description TEXT) AS $proc$
DECLARE
  app TEXT;
BEGIN
  RAISE NOTICE '---';
  RAISE NOTICE 'Setting up % user %', description, username;
  CALL pg_temp.make_user(username);
  FOREACH app in ARRAY ARRAY['admin', 'build_cache', 'build_scans', 'keycloak'] LOOP
    CALL pg_temp.grant_role_to_user(app || '_' || suffix, username);
  END LOOP;
END;
$proc$ LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.setup_app_user(username TEXT) AS $proc$
BEGIN
  CALL pg_temp.setup_user(username, 'user', 'application');
END;
$proc$ LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.setup_migrator_user(username TEXT) AS $proc$
BEGIN
  CALL pg_temp.setup_user(username, 'owner', 'migrator');
END;
$proc$ LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.setup_monitor_user(username TEXT) AS $proc$
BEGIN
  RAISE NOTICE '---';
  RAISE NOTICE 'Setting up monitor user %', username;
  CALL pg_temp.make_user(username);
  CALL pg_temp.grant_role_to_user('pg_read_all_stats', username);
END;
$proc$ LANGUAGE plpgsql;

DO $$
BEGIN
  CALL pg_temp.verify_required_roles();

  -- EDIT THE FOLLOWING LINES TO ALTER DB USERS SET UP
  CALL pg_temp.setup_app_user('ge_app');
  CALL pg_temp.setup_migrator_user('ge_migrator');
  CALL pg_temp.setup_monitor_user('ge_monitor');
  -- END - DO NOT EDIT PAST HERE

  COMMIT;
  RAISE NOTICE '---';
  RAISE NOTICE 'Database users set up successfully';
END;
$$ LANGUAGE plpgsql;
