#!/usr/bin/env bash

set -e
set -o functrace
set -o pipefail

logger() {
    echo "$(date +'%Y-%m-%d %T') - $1";
}

print_help_and_exit() {
    echo "Usage: PGPASSWORD=\"xxx\" [DATABASE_APP_USER_PASSWORD=\"xxx\"] [DATABASE_MIGRATOR_USER_PASSWORD=\"xxx\"] setup.sh --host=my-host.db.com [--port=5432] [--user=postgres] [--database=gradle_enterprise]"
    echo ""
    echo "  Sets up external database for Develocity"
    echo ""
    echo "   --host     Host where the external database is located. Required."
    echo "   --port     Port to connect to. Default 5432."
    echo "   --user     Database user to connect as (must be superuser). Default 'postgres'."
    echo "   --database Database name to connect to. Default 'gradle_enterprise'."
    echo "   --role     Database role to connect as if different to the user. Optional."
    echo ""
    echo "If app / migrator user password is provided it will be set automatically, otherwise user is responsible to setting it on their own."
    echo ""
    echo "   Examples:"
    echo "      PGPASSWORD=\"xxx\" setup.sh --host=my-host.db.com"
    echo ""
    exit "$1"
}

export PGAPPNAME=$(basename "$0")

DB_HOST=
DB_PORT=5432
DB_USER=postgres
DB_NAME=gradle_enterprise
DB_SHARED_SCHEMA=public
DB_ROLE=

# Parse and validate input
for i in "$@"
do
case ${i} in
    --host=*)
    DB_HOST="${i#*=}"
    ;;
    --port=*)
    DB_PORT="${i#*=}"
    ;;
    --user=*)
    DB_USER="${i#*=}"
    ;;
    --role=*)
    DB_ROLE="${i#*=}"
    ;;
    --database=*)
    DB_NAME="${i#*=}"
    ;;
    --shared-schema=*)
    DB_SHARED_SCHEMA="${i#*=}"
    ;;
    *)
	echo "ERROR: Unsupported Parameter ${i}"
	HELP=YES
	HELP_EXIT_CODE=1
    ;;
esac
done

# Print Usage
if [ -n "$HELP" ]; then
  print_help_and_exit ${HELP_EXIT_CODE}
fi

if [ -z "$DB_HOST" ] ; then
  logger "Please specify your database host via the --host parameter"
  print_help_and_exit 1
fi

if [ -z "$PGPASSWORD" ] ; then
  logger "Please specify your database password via the PGPASSWORD environmental variable"
  print_help_and_exit 1
fi

CURRENT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" >/dev/null 2>&1 && pwd )"
SQL_DIR="$CURRENT_DIR/sql"

CONN_PARAMS="-U $DB_USER -h $DB_HOST -p $DB_PORT"
PSQL_PARAMS="$CONN_PARAMS -v ON_ERROR_STOP=1"
PSQL_WITH_GE_DB="psql $PSQL_PARAMS -d $DB_NAME"

if [ -n "$DB_ROLE" ] ; then
  if [ -n "$PGOPTIONS" ] ; then
    PGOPTIONS="$PGOPTIONS "
  fi
  export PGOPTIONS="${PGOPTIONS}-c role=$DB_ROLE"
fi

logger "Setting up database"

# Sanity check - can we connect to the postgres db as the given user?
logger "Testing database connection"
sanity_check_output=$($PSQL_WITH_GE_DB -t -c "SELECT 1234" 2>&1 | xargs echo -n) || true
if [[ $sanity_check_output != "1234" ]] ; then
  logger "Could not connect to the $DB_NAME database at $DB_HOST $DB_PORT: $sanity_check_output"
  exit 1
fi

logger "Creating top-level database objects"
if [[ $DB_SHARED_SCHEMA != "public" ]] ; then
  cat "$SQL_DIR/create_db_objects.sql" \
    | sed "s/CALL pg_temp.setup();/CALL pg_temp.setup('$DB_SHARED_SCHEMA');/" \
    > /tmp/create_db_objects_non_public_shared_schema.sql
  $PSQL_WITH_GE_DB -f /tmp/create_db_objects_non_public_shared_schema.sql
else
  $PSQL_WITH_GE_DB -f "$SQL_DIR/create_db_objects.sql"
fi

logger "Creating database users"
$PSQL_WITH_GE_DB -f "$SQL_DIR/create_db_users.sql"

logger "Setting users passwords"
if [ -n "$DATABASE_APP_USER_PASSWORD" ]; then
  $PSQL_WITH_GE_DB -c "ALTER USER ge_app PASSWORD '$DATABASE_APP_USER_PASSWORD'"
fi

if [ -n "$DATABASE_MIGRATOR_USER_PASSWORD" ]; then
  $PSQL_WITH_GE_DB -c "ALTER USER ge_migrator PASSWORD '$DATABASE_MIGRATOR_USER_PASSWORD'"
fi

logger "Database setup complete"
