#!/usr/bin/env bash

set -e
set -o functrace
set -o pipefail

logger() {
  echo "$(date +'%Y-%m-%d %T') - $1"
}

SOURCE_DB_HOST=
SOURCE_DB_PORT=5432
SOURCE_DB_NAME=gradle_enterprise
SOURCE_DB_USER=dotcom
SOURCE_DB_PASSWORD=dotcom
SOURCE_K8S_NAMESPACE=

DESTINATION_DB_HOST=
DESTINATION_DB_PORT=5432
DESTINATION_DB_USER=postgres
DESTINATION_DB_NAME=gradle_enterprise

INSTALLATION_TYPE=

EXECUTE_CACHE_MIGRATION=true
EXECUTE_SCANS_MIGRATION=false

DATA_DIR="$(pwd)/migrate-from-embedded_$(date +%Y-%m-%d_%H-%M-%S)"

SCRIPT_NAME=$(basename "$0")

print_help_and_exit() {
  echo ""
  echo "Usage: DESTINATION_PASSWORD=\"xxx\" $SCRIPT_NAME --installation-type=kubernetes --destination-host=my-host.db.com [--destination-port=5432] [--destination-user=jdoe] [--destination-database=gradle_enterprise] [--data-dir=/path/to/dir] [--source-kubernetes-namespace=gradle-enterprise]"
  echo ""
  echo "   Migrates admin, account and and configuration data from an embedded Develocity database."
  echo ""
  echo "   --installation-type Installation type. One of kubernetes or openshift. Required."
  echo "   --destination-host Destination database host. Required."
  echo "   --destination-port Destination database port. Default '$DESTINATION_DB_PORT'."
  echo "   --destination-user Destination database user (must be superuser or equivalent). Default '$DESTINATION_DB_USER'."
  echo "   --destination-database Destination database name. Default '$DESTINATION_DB_NAME'."
  echo "   --data-dir Directory to use for temporary database dumps. Default current working directory."
  echo "   --skip-cache-migration Skips cache migration. Use if you want to skip cache migration and configure build cache from scratch."
  echo "   --include-build-scans Includes build scans schema migration. Note: this schema may be huge, use only if you are ok to create a big database dump."
  echo "   --source-kubernetes-namespace Kubernetes namespace of source installation. Required."
  echo ""
  echo "   Examples:"
  echo "      DESTINATION_PASSWORD=\"xxx\" $SCRIPT_NAME --installation-type=kubernetes --destination-host=my-host.db.com --source-kubernetes-namespace=gradle-enterprise"
  echo ""
  exit "$1"
}

# Parse and validate input
for i in "$@"; do
  case ${i} in
  --source-host=*) # not documented, for tests only
    SOURCE_DB_HOST="${i#*=}"
    ;;
  --source-kubernetes-namespace=*)
    SOURCE_K8S_NAMESPACE="${i#*=}"
    ;;
  --destination-host=*)
    DESTINATION_DB_HOST="${i#*=}"
    ;;
  --destination-port=*)
    DESTINATION_DB_PORT="${i#*=}"
    ;;
  --destination-user=*)
    DESTINATION_DB_USER="${i#*=}"
    ;;
  --destination-database=*)
    DESTINATION_DB_NAME="${i#*=}"
    ;;
  --installation-type=*)
    INSTALLATION_TYPE="${i#*=}"
    ;;
  --data-dir=*)
    DATA_DIR="${i#*=}"
    ;;
  --skip-cache-migration)
    EXECUTE_CACHE_MIGRATION=false
    ;;
  --include-build-scans)
    EXECUTE_SCANS_MIGRATION=true
    ;;
  *)
    echo "ERROR: Unsupported Parameter ${i}"
    HELP=YES
    HELP_EXIT_CODE=1
    ;;
  esac
done

# Print Usage
if [ -n "$HELP" ]; then
  print_help_and_exit ${HELP_EXIT_CODE}
fi

command -v sed >/dev/null 2>&1 || {
  echo >&2 "Script requires sed but it's not installed. Please install it and re-run the script. Aborting."
  exit 1
}
command -v psql >/dev/null 2>&1 || {
  echo >&2 "Script requires psql but it's not installed. Please install it and re-run the script. Aborting."
  exit 1
}
command -v pg_dump >/dev/null 2>&1 || {
  echo >&2 "Script requires pg_dump but it's not installed. Please install it and re-run the script. Aborting."
  exit 1
}
command -v pg_restore >/dev/null 2>&1 || {
  echo >&2 "Script requires pg_restore but it's not installed. Please install it and re-run the script. Aborting."
  exit 1
}

PG_DUMP_VERSION=$(pg_dump --version | egrep -o ' [0-9]{1,}\.[0-9]{1,} ?' | xargs)
PG_DUMP_VERSION_MAJOR_ONLY=${PG_DUMP_VERSION%%.*}
if [[ $PG_DUMP_VERSION_MAJOR_ONLY -lt 12 ]]; then
  echo >&2 "Script requires pg_dump version 12 or above, detected version: $PG_DUMP_VERSION_MAJOR_ONLY. Please install supported version and re-run the script. Aborting."
  exit 1
fi

if [ -z "$DESTINATION_DB_HOST" ]; then
  logger "Please specify your database host via --destination-host parameter"
  print_help_and_exit 1
fi

if [ -z "$DESTINATION_PASSWORD" ]; then
  logger "Please specify your database password via the DESTINATION_PASSWORD environmental variable"
  print_help_and_exit 1
fi

if [[ $INSTALLATION_TYPE == "kubernetes" || $INSTALLATION_TYPE == "openshift" ]]; then
  IS_K8S=true
fi

if [[ $INSTALLATION_TYPE == "kubernetes" && -z $SOURCE_K8S_NAMESPACE ]]; then
  logger "Please specify your kubernetes namespace via --source-kubernetes-namespace parameter"
  print_help_and_exit 1
fi

if [[ $INSTALLATION_TYPE == "kubernetes" ]]; then
  K8S_CMD=kubectl
elif [[ $INSTALLATION_TYPE == "openshift" ]]; then
  K8S_CMD=oc
elif [[ $INSTALLATION_TYPE == "test-only-mode" ]]; then # not documented, for tests only
  logger "Warning: Test mode, not for production use."
else
  logger "Invalid value or no value specified for --installation-type parameter"
  print_help_and_exit 1
fi

if [[ -n $SOURCE_K8S_NAMESPACE ]]; then
  K8S_CMD="$K8S_CMD --namespace $SOURCE_K8S_NAMESPACE"
fi

export PGAPPNAME=$(basename "$0")

SOURCE_DB_CONN_PARAMS="-U $SOURCE_DB_USER -h $SOURCE_DB_HOST -p $SOURCE_DB_PORT"
DESTINATION_DB_CONN_PARAMS="-U $DESTINATION_DB_USER -h $DESTINATION_DB_HOST -p $DESTINATION_DB_PORT"

SOURCE_DB_PSQL_PARAMS="$SOURCE_DB_CONN_PARAMS -v ON_ERROR_STOP=1"
DESTINATION_DB_PSQL_PARAMS="$DESTINATION_DB_CONN_PARAMS -v ON_ERROR_STOP=1"

SOURCE_DB_PSQL="psql $SOURCE_DB_PSQL_PARAMS -d $SOURCE_DB_NAME"
DESTINATION_DB_PSQL="psql $DESTINATION_DB_PSQL_PARAMS -d $DESTINATION_DB_NAME"

DESTINATION_DB_URL="$DESTINATION_DB_HOST:$DESTINATION_DB_PORT/$DESTINATION_DB_NAME"

if [[ $IS_K8S == true ]]; then
  SOURCE_DB_POD=$($K8S_CMD get pods --selector=app.kubernetes.io/component=database -o jsonpath='{.items[*].metadata.name}')
  if [[ -z $SOURCE_DB_POD ]]; then
    logger "Could not find a Develocity embedded database pod in namespace '$SOURCE_K8S_NAMESPACE'"
    print_help_and_exit 1
  fi
  logger "Using database pod '$SOURCE_DB_POD'"
fi

# Sanity check - can we connect to the postgres db as the given user?
logger "Testing embedded database connection"
if [[ $IS_K8S == true ]]; then
  embedded_db_sanity_check_output=$($K8S_CMD exec $SOURCE_DB_POD -c database -- sh -c "psql -U $SOURCE_DB_USER $SOURCE_DB_NAME -v ON_ERROR_STOP=1 -t -c 'SELECT 1234' 2>&1 | xargs echo -n") || true
else
  embedded_db_sanity_check_output=$(PGPASSWORD=$SOURCE_DB_PASSWORD $SOURCE_DB_PSQL -t -c "SELECT 1234" 2>&1 | xargs echo -n) || true
fi
if [[ $embedded_db_sanity_check_output != "1234" ]]; then
  logger "Could not connect to source database: $SOURCE_DB_NAME at $SOURCE_DB_HOST:$SOURCE_DB_PORT with user $SOURCE_DB_USER, error: $embedded_db_sanity_check_output"
  exit 1
fi

logger "Testing destination database connection: $DESTINATION_DB_URL"
external_db_sanity_check_output=$(PGPASSWORD=$DESTINATION_PASSWORD $DESTINATION_DB_PSQL -t -c "SELECT 1234" 2>&1 | xargs echo -n) || true
if [[ $external_db_sanity_check_output != "1234" ]]; then
  logger "Could not connect to destination database: $DESTINATION_DB_URL with user $DESTINATION_DB_USER, error:  $external_db_sanity_check_output"
  exit 1
fi

# In case we trap an exit outside of the function
schema='unknown'
trap 'last_command=$current_command; current_command=$BASH_COMMAND' DEBUG
trap 'logger "\"${last_command}\" command failed with exit code $?."; echo "Error migrating $schema schema. Please contact Develocity support."' EXIT

verify_schema() {
  local schema="$1"
  local schema_owner="${schema}_owner"

  logger "Verifying schema '$schema' before migration to $DESTINATION_DB_URL database"

  schema_exists_test=$(PGPASSWORD=$DESTINATION_PASSWORD $DESTINATION_DB_PSQL -t -c "SELECT exists(select schema_name FROM information_schema.schemata WHERE schema_name = '$schema');" 2>&1 | xargs echo -n) || true
  if [[ $schema_exists_test != "t" ]]; then
    logger "No '$schema' schema found, did you run setup.sh script before running migration?, error: $schema_exists_test"
    exit 1
  fi

  schema_owner_test_output=$(PGPASSWORD=$DESTINATION_PASSWORD $DESTINATION_DB_PSQL -t -c "select r.rolname as owner from pg_catalog.pg_namespace s join pg_catalog.pg_roles r on r.oid = s.nspowner where s.nspname = '$schema';" 2>&1 | xargs echo -n) || true
  if [[ $schema_owner_test_output != "$schema_owner" ]]; then
    logger "Schema owner is expected to be '$schema_owner' but was: '$schema_owner_test_output', did you run setup.sh script before running migration?"
    exit 1
  fi

  schema_is_empty_test=$(PGPASSWORD=$DESTINATION_PASSWORD $DESTINATION_DB_PSQL -t -c "SELECT exists(SELECT FROM information_schema.tables WHERE table_schema = '$schema');" 2>&1 | xargs echo -n) || true
  if [[ $schema_is_empty_test == "t" ]]; then
    logger "Schema '$schema' is not empty. Aborting."
    exit 1
  fi
}

# Lock should protect all pg_dump invocations. Only one pg_dump should be running at a time.
PG_DUMP_LOCK_FILE="/var/lock/pg_dump.lock"

wait_for_schema_dump() {
  local schema="$1"
  local container_dump_dir="/opt/gradle/backup/postgres/migrate-from-embedded/$schema"
  # Repeatedly try to acquire the lock with a 5 second timeout. While the lock is held, pg_dump is running.
  # Report progress between these checks by showing the increasing size of the container output directory over time.
  while true; do
    if $K8S_CMD exec "$SOURCE_DB_POD" -c database -- flock -w 5 -x --verbose $PG_DUMP_LOCK_FILE true >/dev/null 2>&1; then
      logger "Finished database dump for schema '$schema'"
      break
    fi
    $K8S_CMD exec "$SOURCE_DB_POD" -c database -- du -sh "$container_dump_dir"
  done
}

dump_schema() {
  local schema="$1"
  local dump_dir="$DATA_DIR/$schema"

  # shellcheck disable=SC2086
  if [[ $IS_K8S == true ]]; then
    # Don't lead with / as this causes a spurious warning from tar in the cp command below.
    # We do want to force it to be in the dir under /opt/gradle though as that's a specific mounted
    # partition, so we re-add the / later just to be safe.
    local container_relative_dir="opt/gradle/backup/postgres/migrate-from-embedded/$schema"
    local container_dump_dir="/$container_relative_dir"

    # If the lock is released and the schema dump output directory is created, then chances are that the dump for this schema
    # has been finished. So skip creating it again.
    if $K8S_CMD exec $SOURCE_DB_POD -c database -- flock -n -x $PG_DUMP_LOCK_FILE -c "ls $container_dump_dir > /dev/null 2>&1" 2>/dev/null; then
      logger "The container output directory for schema '$schema' exists and pg_dump isn't running currently, so it has likely run and completed.
If you need to, you can clear the existing schema dump on the embedded database container by running:
  $K8S_CMD exec '$SOURCE_DB_POD' -c database -- rm -rf '$container_dump_dir'
And then rerunning this script."

    # If a dump is in progress, then await its completion.
    elif [[ $($K8S_CMD exec $SOURCE_DB_POD -c database -- bash -c "pgrep -o -a flock | head | xargs | cut -d' ' -f2-") == "flock -n -x /var/lock/pg_dump.lock -c sh /var/tmp/pg_dump_"* ]]; then
      # shellcheck disable=SC2155
      local schemaForInProgressDump=$($K8S_CMD exec $SOURCE_DB_POD -c database -- bash -c "pgrep -o -a flock | head | cut -d'/' -f7- | xargs | cut -d'_' -f3- | cut -d'.' -f1")
      logger "Dump for schema '$schemaForInProgressDump' is in progress. Waiting for it to complete."
      wait_for_schema_dump $schemaForInProgressDump

    # If the lock is released, then execute pg_dump. To have reached this point, we must have already checked that the dump wasn't
    # previously created, and isn't in progress.
    elif $K8S_CMD exec $SOURCE_DB_POD -c database -- flock -n -x $PG_DUMP_LOCK_FILE true 2>/dev/null; then
      logger "Dumping source database schema '$schema' before migration to $DESTINATION_DB_URL database"

      # This script contains the 'critical section' that is protected by flock.
      # It cleans the state managed by the pg_dump invocation (output dir, logs), and invokes pg_dump.
      local pg_dump_script="$DATA_DIR/pg_dump_$schema.sh"
      local container_pg_dump_script="/var/tmp/pg_dump_$schema.sh"
      local container_pg_dump_log_file="/var/tmp/pg_dump_$schema.log"
      mkdir -p $DATA_DIR
      cat <<EOF >"$pg_dump_script"
rm -rf $container_dump_dir
mkdir -p $container_dump_dir
rm -f $container_pg_dump_log_file
pg_dump -U $SOURCE_DB_USER --schema="$schema" --format=directory --jobs=4 --file="$container_dump_dir" "$SOURCE_DB_NAME" > "$container_pg_dump_log_file" 2>&1
EOF
      $K8S_CMD cp -c database $pg_dump_script $SOURCE_DB_POD:$container_pg_dump_script

      # Run the script, underneath the flock-managed mutex. Do this in the background. Once the script exits, it will automatically release the lock.
      # If the lock is not free, flock will fail due to the fact that
      #  (a) The lock is exclusive (-x)
      #  (b) The acquire in this command is non-blocking (-n), which means that it will fail rather than wait for the lock to be available.
      $K8S_CMD exec -it $SOURCE_DB_POD -c database -- nohup sh -c "flock -n -x $PG_DUMP_LOCK_FILE -c 'sh $container_pg_dump_script' &" > /dev/null 2>&1

      # Having initiated pg_dump, poll for its completion.
      wait_for_schema_dump $schema
    else
      logger "Database migration failed: Schema dump for '$schema' could not be created."
      exit 1
    fi

    # Polling has finished, so pg_dump is no longer running for this schema.
    # If the local output directory exists, then assume that pg_dump has run, and it has been copied locally, so we don't need to do anything.
    if [[ -d $dump_dir ]]; then
      logger "Skipping copying of dump for schema '$schema' from container to local machine.
The output directory '$dump_dir' already exists, so it has likely been copied previously.
If you need to, you can clear the previous dump from the local machine to copy it again by running:
  rm -rf $dump_dir
And then rerunning this script."
    else
      # If the local output directory doesn't exist, and the lock has been released, then either:
      #  (a) We kicked off pg_dump on the container in this script execution, and it finished and released the lock, and now we're here.
      #  (b) We kicked off pg_dump on the container in a previous script, execution, then in this script execution we polled for it until
      #      it finished and released the lock, and now we're here.
      # In both cases, the local directory does not exist, and we have reached a point in the script where the container output does exist
      # and is ready to be copied locally. So, we copy the container dump to the local machine.
      logger "Copying dumped '$schema' schema from source database container's backup volume into local directory"
      $K8S_CMD cp -c database $SOURCE_DB_POD:$container_relative_dir "$dump_dir"
      logger "Schema '$schema' was successfully dumped locally."
    fi
  else
    logger "Dumping source database schema '$schema' before migration to $DESTINATION_DB_URL database"
    rm -rf "$dump_dir"
    mkdir -p "$dump_dir"
    PGPASSWORD=$SOURCE_DB_PASSWORD pg_dump $SOURCE_DB_CONN_PARAMS --schema="$schema" --format=directory --jobs=4 --file="$dump_dir" "$SOURCE_DB_NAME"
    logger "Schema '$schema' was successfully dumped locally."
  fi
}

restore_schema() {
  local schema="$1"
  local schema_owner="${schema}_owner"
  local dump_dir="$DATA_DIR/$schema"

  logger "Restoring data into destination database $DESTINATION_DB_URL '$schema' schema"

  # shellcheck disable=SC2086
  PGPASSWORD=$DESTINATION_PASSWORD pg_restore $DESTINATION_DB_CONN_PARAMS -d "$DESTINATION_DB_NAME" --schema="$schema" --no-owner --role="$schema_owner" --format=directory --jobs=4 "$dump_dir"

  logger "Schema '$schema' was successfully restored to $DESTINATION_DB_URL."
}

verify_schema admin
verify_schema keycloak

if [[ $EXECUTE_CACHE_MIGRATION == true ]]; then
  verify_schema build_cache
fi

if [[ $EXECUTE_SCANS_MIGRATION == true ]]; then
  verify_schema build_scans
fi

dump_schema admin
dump_schema keycloak

if [[ $EXECUTE_CACHE_MIGRATION == true ]]; then
  dump_schema build_cache
fi

if [[ $EXECUTE_SCANS_MIGRATION == true ]]; then
  dump_schema build_scans
fi

restore_schema admin
restore_schema keycloak

if [[ $EXECUTE_CACHE_MIGRATION == true ]]; then
  restore_schema build_cache
fi

if [[ $EXECUTE_SCANS_MIGRATION == true ]]; then
  restore_schema build_scans
fi

if [[ $IS_K8S == true ]]; then
  $K8S_CMD exec "$SOURCE_DB_POD" -c database -- rm -rf /opt/gradle/backup/postgres/migrate-from-embedded
fi
rm -rf "$DATA_DIR"

trap - EXIT

logger "Database migration complete $DESTINATION_DB_URL"
