-- Create local functions to use. Stuff in pg_temp is session-local and disappears
-- at disconnection.

-- There are a few places where we test for existence and then do an IF NOT EXISTS.
-- The aim here is to a) give the best meaningful message, and we can only give
-- a message indicating that something was created if we looked for it first, and
-- b) we don't want to spam the server logs by omitting the IF NOT EXISTS, since that
-- will log errors to the server log, even if we catch them here. So we do both, and
-- will only show both our "thing created" message and the automatic NOTICE that we
-- typically get if something already existed in the event of a race condition, which
-- is fairly unlikely.

CREATE PROCEDURE pg_temp.create_extension(ename TEXT, target_schema TEXT, optional BOOLEAN = false) AS $proc$
BEGIN
  IF EXISTS (SELECT FROM pg_available_extensions WHERE name = ename) THEN
    IF NOT EXISTS (SELECT FROM pg_extension WHERE pg_extension.extname = ename) THEN
      EXECUTE format('CREATE EXTENSION IF NOT EXISTS %I SCHEMA %I', quote_ident(ename), quote_ident(target_schema));
      RAISE NOTICE 'Installed Extension %', ename;
    ELSE
      RAISE NOTICE 'Extension % already installed', ename;
    END IF;
  ELSIF optional THEN
    RAISE NOTICE 'Optional extension % is not available', ename;
  ELSE
    RAISE EXCEPTION 'Required extension % is not available', ename;
  END IF;
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.create_role(rname TEXT) AS $proc$
BEGIN
  IF NOT EXISTS (SELECT FROM pg_roles WHERE rolname = rname) THEN
    BEGIN
      EXECUTE format('CREATE ROLE %I NOLOGIN', quote_ident(rname));
      RAISE NOTICE 'Created role %', rname;
    EXCEPTION WHEN duplicate_object THEN
      RAISE NOTICE 'Role % already created', rname;
    END;
  ELSE
    RAISE NOTICE 'Role % already created', rname;
  END IF;
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.create_schema(sname TEXT, owner TEXT) AS $proc$
BEGIN
  IF NOT EXISTS (SELECT FROM pg_namespace WHERE nspname = sname) THEN
    BEGIN
      EXECUTE format('CREATE SCHEMA IF NOT EXISTS %I', quote_ident(sname));
      EXECUTE format('ALTER SCHEMA %I OWNER TO %I', quote_ident(sname), quote_ident(owner));
      RAISE NOTICE 'Created schema %', sname;
    EXCEPTION WHEN duplicate_object THEN
      RAISE NOTICE 'Schema % already created', sname;
    END;
  ELSE
    RAISE NOTICE 'Schema % already created', sname;
  END IF;

  IF owner <> (SELECT nspowner::regrole::text FROM pg_namespace WHERE nspname = sname) THEN
    EXECUTE format('ALTER SCHEMA %I OWNER TO %I', quote_ident(sname), quote_ident(owner));
    RAISE NOTICE 'Altered schema % owner to %', sname, owner;
  END IF;
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.setup_app(appname TEXT, schema_suffix TEXT DEFAULT '') AS $proc$
DECLARE
  owner TEXT = appname || '_owner';
  app_user TEXT = appname || '_user';
  schema_name TEXT = appname || schema_suffix;
  session_user TEXT = (SELECT session_user);
BEGIN
  CALL pg_temp.create_role(owner);
  CALL pg_temp.create_role(app_user);

  EXECUTE format('GRANT %I TO %I', owner, session_user); -- necessary to be able to assign that role as schema owner in RDS environments
  CALL pg_temp.create_schema(schema_name, owner);

  -- set role necessary in AWS Aurora as in some circumstances a db owner can't GRANT on a schema owned by another role
  EXECUTE format('SET ROLE TO %I', owner);
  EXECUTE format('GRANT ALL ON SCHEMA %I TO %I', quote_ident(schema_name), quote_ident(app_user));
  RESET ROLE;

  EXECUTE format('REVOKE %I FROM %I', owner, session_user); -- role is no longer needed
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.create_extensions(target_schema TEXT DEFAULT 'public') AS $proc$
BEGIN
  RAISE NOTICE 'Creating extensions';
  CALL pg_temp.create_extension('pgcrypto', target_schema);
  CALL pg_temp.create_extension('intarray', target_schema);
  CALL pg_temp.create_extension('amcheck', target_schema, true);
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.setup_apps(schema_suffix TEXT DEFAULT '') AS $proc$
BEGIN
  RAISE NOTICE 'Creating roles and schemas';
  CALL pg_temp.setup_app('admin', schema_suffix);
  CALL pg_temp.setup_app('build_cache', schema_suffix);
  CALL pg_temp.setup_app('build_scans', schema_suffix);
  CALL pg_temp.setup_app('keycloak', schema_suffix);

  -- build scans app needs some extra perms currently, mostly related to triggering vacuums
  IF EXISTS (SELECT FROM pg_roles WHERE rolname = 'build_scans_owner') THEN
    IF NOT EXISTS (SELECT FROM pg_auth_members WHERE roleid = 'build_scans_owner'::regrole AND member = 'build_scans_user'::regrole) THEN
      GRANT build_scans_owner TO build_scans_user;
    END IF;
  END IF;
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.setup_privileged_functions(target_schema TEXT DEFAULT 'public') AS $proc$
BEGIN
  RAISE NOTICE 'Setting up privileged functions';

  EXECUTE format($exec$
    CREATE OR REPLACE FUNCTION %I.setup_auto_explain(log_min_duration TEXT, log_format TEXT DEFAULT 'json', log_level TEXT DEFAULT 'info') RETURNS VOID AS $func$
    LOAD 'auto_explain';
    SELECT set_config('auto_explain.log_min_duration', log_min_duration, false);
    SELECT set_config('auto_explain.log_format', log_format, false);
    SELECT set_config('auto_explain.log_level', log_level, false);
    $func$ LANGUAGE SQL SECURITY DEFINER;
  $exec$, quote_ident(target_schema));

  EXECUTE format($exec$
    CREATE OR REPLACE FUNCTION %I.setup_local_nested_auto_explain(log_min_duration TEXT) RETURNS VOID AS $func$
    SELECT set_config('auto_explain.log_min_duration', log_min_duration, true);
    SELECT set_config('auto_explain.log_nested_statements', 'true', true);
    $func$ LANGUAGE SQL SECURITY DEFINER;
  $exec$, quote_ident(target_schema));

  BEGIN
    EXECUTE format($exec$
      GRANT EXECUTE ON FUNCTION %I.bt_index_check(regclass) TO build_scans_user;
    $exec$, quote_ident(target_schema));
  EXCEPTION WHEN OTHERS THEN
    -- GCS doesn't have amcheck available, but allows the CREATE EXTENSION command above
    -- to pass with a warning. This will fail though - just ignore it.
    RAISE WARNING 'Unable to grant permission on %.bt_index_check(regclass) - %', target_schema, SQLSTATE;
  END;

  BEGIN
    GRANT EXECUTE ON FUNCTION pg_rotate_logfile() TO build_scans_user;
  EXCEPTION WHEN OTHERS THEN
    -- some environments like RDS don't allow us to do this, but we don't care about the
    -- log file in these cases
    RAISE WARNING 'Unable to grant permission on pg_rotate_logfile() - %', SQLSTATE;
  END;
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.setup_shared_schema(target_schema TEXT DEFAULT 'public') AS $proc$
BEGIN
  IF target_schema <> 'public' THEN
    RAISE NOTICE 'Setting up shared schema';
    CALL pg_temp.create_schema(target_schema, current_user);
    EXECUTE format('GRANT USAGE ON SCHEMA %I TO PUBLIC', quote_ident(target_schema));
  END IF;
END;
$proc$
LANGUAGE plpgsql;

CREATE PROCEDURE pg_temp.setup(target_schema TEXT DEFAULT 'public') AS $proc$
BEGIN
  CALL pg_temp.setup_shared_schema(target_schema);

  RAISE NOTICE '---';
  CALL pg_temp.create_extensions(target_schema);

  RAISE NOTICE '---';
  CALL pg_temp.setup_apps();

  RAISE NOTICE '---';
  CALL pg_temp.setup_privileged_functions(target_schema);

  COMMIT;
  RAISE NOTICE '---';
  RAISE NOTICE 'Database objects set up successfully';
END;
$proc$
LANGUAGE plpgsql;

DO $$
BEGIN
  CALL pg_temp.setup();
END;
$$ LANGUAGE plpgsql;
